% 2D VERTEX MODEL OF DROSOPHILA VENTRAL FURROW FORMATION
% MATLAB version with utilization of mex compilers
% By Oleg Polyakov | Princeton University | 2008 - 2013
% Modified in order to test the role of ectodermal compression 
% Modified by Hanqing Guo and Bing He

% This model was developed to study the process of ventral furrow formation
% in Drosophila Melanogaster.

% In this model we consider the consider the midplane cross section of the
% emrbyo at the onset of gasrtulation. This corresponds to a geometry of 80
% cells connected in a ring. The volumes of the yolk, the cellular
% volumes, as well as the apical, basal, and lateral sides of all the cells
% have an "elastic" energy equation of the form K*(X - Xo)^2.

% The input energy comes entirely from apical constriction. It is a
% Gaussian distribution centered on the ventral surface with a width of
% sigma.

% The supporting mex files are the following:
% INIT_C : initializes the cross sectional geometry, the equilibrium values
% of the cells and apical constriction

% CALC_GRAD_C: calculates the gradient by taking the derivative of the
% energy equation

% CALC_GRAD_C_N_STEPS: calculates the gradient and moves the verticies
% along the gradient with a magnitude of P.beta. This is repeated P.N_steps
% times. If equilibrium is reached then the flag P.Done is set to 0.

% GEOM_OUT_C: outputs the current geometry parameters for each cell.


%% Input parameters
clearvars

%Fraction of ectoderm shortening
pl = 0.8; % pl = 1 means there is no ectodermal compression

%scaling factor for P.KL in the constricting cells.
%P.KL is the passive lateral spring constant.
sf = 1; % smaller passive lateral force in the constricting cells when sf<1

% active lateral spring constant in the constricting cells
L_Alpha = 2;

% Set up apical constriction mutant
AC_defect = 0; % "0" for wildtype, "1" for constriction mutant (constriction in every other cell)

% Set up optogenetic inhibition of apical contractility
stimulation = 2;
% Note: "stimulation" defines the intermediate stage when apical contrictile force is eliminated
% No stimulation will be imposed if stimulation > 11

%% Init
% Flag to see when simulation comes to a stop
P.Done = 1;

% Outside and inner radius of the emrbyo
P.R_out = 14; P.R_in = 8.9;

% Apical constriction
P.Alpha = [0, 5000, 0]; % [apical linear(0,not used), apical quadratic(used), basal linear(0,not used)]
P.S = [1, 3, 1]; % % sigma for P.Alpha P.S(2) for WT is 3. Reduce P.S(2) to reduce the size of apical constriction domain 

KL_val = 20.0;
KA_val = 30.0;
KB_val = 1000.0;
KV_val = 5000;
KY_val = [1, -0]; %[yolk compressibility, yolk pressure(0,not used)]

% Simulation runs - define step size and max steps
P.N_steps = 100000; P.beta = .0025/KV_val; P.N_steps_end = 10000-1;

% Inititialize values
[Ax, Ay, Bx, By, P.Ao_con, P.Ao, P.KA_lin, P.KA_con, P.KA, P.Bo, P.KB_lin, P.KB,...
    P.Lo, P.KL, P.Vo, P.KV, P.Yo, P.KY] = INIT_C(P.R_out, P.R_in, P.Alpha, P.S,...
    KA_val, KB_val, KL_val, KV_val, KY_val);

% Set up the distribution of L0 for the apical constriction springs
% ventral mesoderm: 0; lateral mesoderm: transition; ectoderm: 1
n_meso_Ao_drop = 12;K_meso_Ao_drop = 7;
P.Ao_con = P.Ao_con.*(1 - 1./(1 + (((1:80)-40.5)./K_meso_Ao_drop).^n_meso_Ao_drop));

%% Init Gradients

% Define the stepwise reduction of KB from 2^10 to 2^0
KB_all = [];
for i = [10:-1:0]
    KB_all = [KB_all, 2^i];
end

%Reset the resting length of the lateral membrane in the ectoderm
n_meso_Lo_drop = 12;K_meso_Lo_drop = 11;
P.Lo = P.Lo.*(1-(1-pl).*(1 - 1./(1 + (((1:80)-41)./K_meso_Lo_drop).^n_meso_Lo_drop)));

%Set the resting length of the active lateral springs
for i = [1:80]
    P.Lo_con(i) = 0;
end

%Redefine lateral passive force for all cells based on "sf"

n_meso_KL_drop = 12;K_meso_KL_drop = 7;
P.KL = P.KL.*(1 - 1./(1 + (((1:80)-41)./K_meso_KL_drop).^n_meso_KL_drop));
P.KL = sf.*KL_val+(1-sf).*P.KL;

%% Define save path

k = 1;
while(k >0)
    savefile_path = [pwd, '/Simulation', num2str(k),'/'];
    if exist(savefile_path)
        k = k + 1;
    else
        break;
    end
end

mkdir(savefile_path);

%% Simulation

N_states = length(KB_all); %number of intermediate states defined by Kb.

for hh = 1:N_states
    
    % define the active lateral spring constant
    n_meso_KL_con = 12;K_meso_KL_con = 7;
    P.KL_con = L_Alpha*(1-(1 - 1./(1 + (((1:80)-41)./K_meso_KL_con).^n_meso_KL_con)));
    
    % define basal spring constant - stepwise reduction
    KB_val = KB_all(hh);
    
    for i = [1:80]
        P.KB(i) = KB_val;
        P.KB_lin(i) = 0;
    end
    
    if AC_defect == 1
        %re-define cells that undergo apcial constriction
        for i = [35:2:46] %constriction happens in every other cell in the ventral domain
            P.KA_con(i) = 0;
        end
    end
    
    if hh>=stimulation %"stimulation" defines the intermediate step when apical contrictile force is eliminated
        % Apical constriction strength redefined
        P.Alpha = [0, 0, 0]; % apical active spring is removed
        P.S = [1, 3, 1]; % sigma for P.Alpha
        
        %Redefines P.KA_lin and P.KA_con based on modified P.Alpha and P.S
        for i = 1:80
            theta = (-pi*P.R_out + (i-1)*2*pi*P.R_out/79.0)/P.S(1);
            P.KA_lin(i) = P.Alpha(1)*exp(-theta*theta/2); % zero, not used.
            
            theta = (-pi*P.R_out + (i-1)*2*pi*P.R_out/79.0)/P.S(2);
            P.KA_con(i) = P.Alpha(2)*exp(-theta*theta/2);
        end
        
        L_Alpha = 0;
        n_meso_KL_con = 12;K_meso_KL_con = 7;
        P.KL_con = L_Alpha*(1-(1 - 1./(1 + (((1:80)-41)./K_meso_KL_con).^n_meso_KL_con)));
        
    end
    
    % simulation
    P.Done = 1;
    Vxm = cell(1);
    Vym = cell(1);
    Gxm = cell(1);
    Gym = cell(1);
    P.A = [];
    P.B = [];
    P.L = [];
    P.V = [];
    P.Y = [];
    P.count = 0; P.loop_count = 0;
    
    % Initiate gradient
    [Gax, Gay, Gbx, Gby, E_old] = CALC_GRAD_C(Ax, Ay, Bx, By,...
        P.KA_lin, P.KA_con, P.Ao_con, P.KA, P.Ao, P.KL_con, P.Lo_con, P.KL, P.Lo, P.KB, P.KB_lin, P.Bo, P.KV, P.Vo,...
        P.KY, P.Yo, P.R_out, P.R_out, P.beta);
    
    P.count = P.count + 1;
    [P.A(P.count, :), P.B(P.count, :), P.L(P.count, :), P.V(P.count, :), P.Y(P.count, :)] = GEOM_OUT_C(Ax, Ay, Bx, By);
    Vxm{P.count} = [Ax;Bx]; Vym{P.count} = [Ay;By]; Gxm{P.count} = [Gax;Gbx]; Gym{P.count} = [Gay;Gby];
    
    % Run simulation until equilibrium is reached
    
    while (P.Done)
        [Gax, Gay, Gbx, Gby, P.Done] = CALC_GRAD_C_N_STEPS(Ax, Ay, Bx, By,...
            P.KA_lin, P.KA_con, P.Ao_con, P.KA, P.Ao, P.KL_con, P.Lo_con, P.KL, P.Lo, P.KB, P.KB_lin, P.Bo, P.KV, P.Vo,...
            P.KY, P.Yo, P.R_out, P.R_out, P.beta, P.N_steps, E_old); P.loop_count = P.loop_count + 1;
        
        disp(['Elongation is ', num2str(P.L(end, 40)/5.1), '  Volume change is ', num2str(min(P.V(end, :)/P.Vo(1)))]);
        P.count = P.count + 1;
        [P.A(P.count, :), P.B(P.count, :), P.L(P.count, :), P.V(P.count, :), P.Y(P.count, :)] = GEOM_OUT_C(Ax, Ay, Bx, By);
        Vxm{P.count} = [Ax;Bx]; Vym{P.count} = [Ay;By]; Gxm{P.count} = [Gax;Gbx]; Gym{P.count} = [Gay;Gby];
        
        if P.loop_count == P.N_steps_end, break; end
    end
    
    P.t_stop = P.loop_count;
    
    %% File to save the data
    
    savefile = [savefile_path, 'Stage', num2str(hh),'.mat'];
    
    if exist(savefile, 'file'), delete(savefile); end
    save(savefile, 'Vxm', 'Vym', 'Gxm', 'Gym', 'P',...
        'AC_defect', 'stimulation', 'pl', 'KB_all', 'N_states');
    
    disp(['Stage ', num2str(hh), ' complete.']);
    
end

